﻿using System.Linq;
using System.Reflection;
using System.ServiceModel;

using HIPS.Web.Components.Cache;
using HIPS.Web.Components.ServiceModel;

namespace HIPS.Web.Data.Hips
{

    /// <summary>
    /// Provides a base class for repositories.
    /// </summary>
    /// <history>
    ///   <change user="David Sampson (Chamonix)" date="28 November 2013">Initial version.</change>
    /// </history>
    public abstract class RepositoryBase<TClientProxy> where TClientProxy : ICommunicationObject
    {

        #region Fields

        /// <summary>
        /// Format string for cache keys to be employed by this repository.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="28 November 2013">Initial version.</change>
        /// </history>
        private readonly string CacheKeyFormat;
        
        #endregion

        #region Properties

        /// <summary>
        /// Gets or sets the client proxy used by the repository.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="28 November 2013">Initial version.</change>
        /// </history>
        protected TClientProxy ClientProxy { get; set; }

        /// <summary>
        /// Gets or sets the cache provider used by the repository.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="28 November 2013">Initial version.</change>
        /// </history>
        protected ICacheProvider CacheProvider { get; set; }

        #endregion

        #region Constructors

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="28 November 2013">Initial version.</change>
        /// </history>
        protected RepositoryBase(ICacheProvider cacheProvider, string cacheKeyPrefix = "")
        {
            this.CacheProvider = cacheProvider;
            this.CacheKeyFormat = string.Format("{0}.{1}.{{0}}", cacheKeyPrefix, this.GetType().Name);
        }

        #endregion

        #region Methods

        /// <summary>
        /// Get the full cache key (including prefixes and a specific value).
        /// </summary>
        /// <param name="cacheKeyValue">Specific value component of the cache key.</param>
        /// <returns>String containing the full cache key.</returns>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="28 November 2013">Initial version.</change>
        /// </history>
        protected string GetFullCacheKey(string cacheKeyValue)
        {
            return string.Format(this.CacheKeyFormat, cacheKeyValue);
        }

        /// <summary>
        /// Gets a full cache key based on a method invocation.
        /// </summary>
        /// <param name="methodName">Name of the method invoked.</param>
        /// <param name="methodParameters">Parameter values specific to the method invocation.</param>
        /// <returns>String containing the full cache key.</returns>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="28 November 2013">Initial version.</change>
        /// </history>
        protected string GetMethodInvocationFullCacheKey(string methodName, params object[] methodParameters)
        {
            string methodParametersString = string.Empty;
            if (methodParameters != null)
            {
                methodParametersString = string.Join(",", methodParameters.Select(p => string.Format("{0}={1}", p == null ? "null" : p.GetType().Name, p == null ? "null": p)));
            }
            return this.GetFullCacheKey(string.Format("{0}({1})", methodName, methodParametersString));
        }

        #region IDisposable

        /// <summary>
        /// Performs application-defined tasks associated with freeing, releasing, or resetting unmanaged resources.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="28 November 2013">Initial version.</change>
        /// </history>
        public void Dispose()
        {
            if (this.ClientProxy != null)
            {
                this.ClientProxy.DisposeSafely();
                this.ClientProxy = default(TClientProxy);
            }
        }

        #endregion

        #endregion

    }

}
